/*
 * Decompiled with CFR 0.152.
 */
package org.jackhuang.hmcl.util.io;

import com.google.errorprone.annotations.CanIgnoreReturnValue;
import java.awt.Desktop;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.glavo.chardet.DetectedCharset;
import org.glavo.chardet.UniversalDetector;
import org.jackhuang.hmcl.util.StringUtils;
import org.jackhuang.hmcl.util.function.ExceptionalConsumer;
import org.jackhuang.hmcl.util.io.DirectoryStructurePrinter;
import org.jackhuang.hmcl.util.logging.Logger;
import org.jackhuang.hmcl.util.platform.OperatingSystem;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public final class FileUtils {
    private static final Set<String> INVALID_WINDOWS_RESOURCE_BASE_NAMES = Set.of("aux", "con", "nul", "prn", "clock$", "com1", "com2", "com3", "com4", "com5", "com6", "com7", "com8", "com9", "com\u00b9", "com\u00b2", "com\u00b3", "lpt1", "lpt2", "lpt3", "lpt4", "lpt5", "lpt6", "lpt7", "lpt8", "lpt9", "lpt\u00b9", "lpt\u00b2", "lpt\u00b3");

    private FileUtils() {
    }

    @Nullable
    public static Path toPath(@Nullable File file) {
        try {
            return file != null ? file.toPath() : null;
        }
        catch (InvalidPathException e) {
            Logger.LOG.warning("Invalid path: " + String.valueOf(file));
            return null;
        }
    }

    @Nullable
    public static List<Path> toPaths(@Nullable List<File> files) {
        if (files == null) {
            return null;
        }
        return files.stream().map(FileUtils::toPath).filter(Objects::nonNull).toList();
    }

    public static boolean canCreateDirectory(String path) {
        try {
            return FileUtils.canCreateDirectory(Paths.get(path, new String[0]));
        }
        catch (InvalidPathException e) {
            return false;
        }
    }

    public static boolean canCreateDirectory(Path path) {
        if (Files.isDirectory(path, new LinkOption[0])) {
            return true;
        }
        if (Files.exists(path, new LinkOption[0])) {
            return false;
        }
        Path lastPath = path;
        for (path = path.getParent(); path != null && !Files.exists(path, new LinkOption[0]); path = path.getParent()) {
            lastPath = path;
        }
        if (path == null) {
            return false;
        }
        if (!Files.isDirectory(path, new LinkOption[0])) {
            return false;
        }
        try {
            Files.createDirectory(lastPath, new FileAttribute[0]);
            Files.delete(lastPath);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public static String getNameWithoutExtension(String fileName) {
        return StringUtils.substringBeforeLast(fileName, '.');
    }

    public static String getNameWithoutExtension(Path file) {
        return StringUtils.substringBeforeLast(FileUtils.getName(file), '.');
    }

    public static String getExtension(String fileName) {
        return StringUtils.substringAfterLast(fileName, '.');
    }

    public static String getExtension(Path file) {
        return StringUtils.substringAfterLast(FileUtils.getName(file), '.');
    }

    public static String normalizePath(String path) {
        return StringUtils.addPrefix(StringUtils.removeSuffix(path, "/", "\\"), "/");
    }

    public static String getName(Path path) {
        Path fileName = path.getFileName();
        return fileName != null ? fileName.toString() : "";
    }

    public static Path toAbsolute(Path path) {
        return path.toAbsolutePath().normalize();
    }

    public static String getAbsolutePath(Path path) {
        return path.toAbsolutePath().normalize().toString();
    }

    public static boolean isNameValid(String name) {
        return FileUtils.isNameValid(OperatingSystem.CURRENT_OS, name);
    }

    public static boolean isNameValid(OperatingSystem os, String name) {
        if (name.isEmpty()) {
            return false;
        }
        if (name.equals(".") || name.equals("..") || name.equals("~")) {
            return false;
        }
        for (int i = 0; i < name.length(); ++i) {
            int codePoint;
            int ch = name.charAt(i);
            if (Character.isSurrogate((char)ch)) {
                char ch2;
                if (!Character.isHighSurrogate((char)ch)) {
                    return false;
                }
                if (i == name.length() - 1) {
                    return false;
                }
                if (!Character.isLowSurrogate(ch2 = name.charAt(++i))) {
                    return false;
                }
                codePoint = Character.toCodePoint((char)ch, ch2);
            } else {
                codePoint = ch;
            }
            if (!Character.isValidCodePoint(codePoint) || Character.isISOControl(codePoint) || codePoint == 47 || codePoint == 0 || codePoint == 65533 || codePoint == 65534 || codePoint == 65535) {
                return false;
            }
            if (os != OperatingSystem.WINDOWS || ch != 60 && ch != 62 && ch != 58 && ch != 34 && ch != 92 && ch != 124 && ch != 63 && ch != 42) continue;
            return false;
        }
        if (os == OperatingSystem.WINDOWS) {
            char lastChar = name.charAt(name.length() - 1);
            if (lastChar == '.') {
                return false;
            }
            if (Character.isWhitespace(lastChar)) {
                return false;
            }
            String basename = StringUtils.substringBeforeLast(name, '.');
            if (INVALID_WINDOWS_RESOURCE_BASE_NAMES.contains(basename.toLowerCase(Locale.ROOT))) {
                return false;
            }
        }
        return true;
    }

    public static long size(Path file) {
        try {
            return Files.size(file);
        }
        catch (NoSuchFileException ignored) {
            return 0L;
        }
        catch (IOException e) {
            Logger.LOG.warning("Failed to get file size of " + String.valueOf(file), e);
            return 0L;
        }
    }

    public static String readTextMaybeNativeEncoding(Path file) throws IOException {
        byte[] bytes = Files.readAllBytes(file);
        if (OperatingSystem.NATIVE_CHARSET == StandardCharsets.UTF_8) {
            return new String(bytes, StandardCharsets.UTF_8);
        }
        UniversalDetector detector = new UniversalDetector();
        detector.handleData(bytes);
        detector.dataEnd();
        DetectedCharset detectedCharset = detector.getDetectedCharset();
        if (detectedCharset != null && detectedCharset.isSupported() && (detectedCharset == DetectedCharset.UTF_8 || detectedCharset == DetectedCharset.US_ASCII)) {
            return new String(bytes, StandardCharsets.UTF_8);
        }
        return new String(bytes, OperatingSystem.NATIVE_CHARSET);
    }

    public static void deleteDirectory(Path directory) throws IOException {
        if (!Files.exists(directory, new LinkOption[0])) {
            return;
        }
        if (!Files.isSymbolicLink(directory)) {
            FileUtils.cleanDirectory(directory);
        }
        Files.deleteIfExists(directory);
    }

    public static boolean deleteDirectoryQuietly(Path directory) {
        try {
            FileUtils.deleteDirectory(directory);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public static void setExecutable(Path path) {
        PosixFileAttributeView view = Files.getFileAttributeView(path, PosixFileAttributeView.class, new LinkOption[0]);
        if (view != null) {
            try {
                Set<PosixFilePermission> oldPermissions = view.readAttributes().permissions();
                if (oldPermissions.contains((Object)PosixFilePermission.OWNER_EXECUTE)) {
                    return;
                }
                EnumSet<PosixFilePermission> permissions = EnumSet.noneOf(PosixFilePermission.class);
                permissions.addAll(oldPermissions);
                permissions.add(PosixFilePermission.OWNER_EXECUTE);
                view.setPermissions(permissions);
            }
            catch (IOException e) {
                Logger.LOG.warning("Failed to set permissions for " + String.valueOf(path), e);
            }
        }
    }

    public static void copyDirectory(Path src, Path dest) throws IOException {
        FileUtils.copyDirectory(src, dest, path -> true);
    }

    public static void copyDirectory(final Path src, final Path dest, final Predicate<String> filePredicate) throws IOException {
        Files.walkFileTree(src, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (!filePredicate.test(src.relativize(file).toString())) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                Path destFile = dest.resolve(src.relativize(file).toString());
                Files.copy(file, destFile, StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (!filePredicate.test(src.relativize(dir).toString())) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                Path destDir = dest.resolve(src.relativize(dir).toString());
                Files.createDirectories(destDir, new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static boolean hasKnownDesktop() {
        if (!OperatingSystem.CURRENT_OS.isLinuxOrBSD()) {
            return true;
        }
        String desktops = System.getenv("XDG_CURRENT_DESKTOP");
        if (desktops == null) {
            desktops = System.getenv("XDG_SESSION_DESKTOP");
        }
        if (desktops == null) {
            return false;
        }
        for (String desktop : desktops.split(":")) {
            switch (desktop.toLowerCase(Locale.ROOT)) {
                case "gnome": 
                case "xfce": 
                case "kde": 
                case "mate": 
                case "deepin": 
                case "x-cinnamon": {
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean moveToTrash(Path file) {
        if (OperatingSystem.CURRENT_OS.isLinuxOrBSD() && FileUtils.hasKnownDesktop()) {
            if (!Files.exists(file, new LinkOption[0])) {
                return false;
            }
            String xdgData = System.getenv("XDG_DATA_HOME");
            Path trashDir = StringUtils.isNotBlank(xdgData) ? Paths.get(xdgData, "Trash") : Paths.get(System.getProperty("user.home"), ".local/share/Trash");
            Path infoDir = trashDir.resolve("info");
            Path filesDir = trashDir.resolve("files");
            try {
                Files.createDirectories(infoDir, new FileAttribute[0]);
                Files.createDirectories(filesDir, new FileAttribute[0]);
                String name = FileUtils.getName(file);
                Path infoFile = infoDir.resolve(name + ".trashinfo");
                Path targetFile = filesDir.resolve(name);
                int n = 0;
                while (Files.exists(infoFile, new LinkOption[0]) || Files.exists(targetFile, new LinkOption[0])) {
                    infoFile = infoDir.resolve(name + "." + ++n + ".trashinfo");
                    targetFile = filesDir.resolve(name + "." + n);
                }
                String time = DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(ZonedDateTime.now().truncatedTo(ChronoUnit.SECONDS));
                if (Files.isDirectory(file, new LinkOption[0])) {
                    FileUtils.copyDirectory(file, targetFile);
                } else {
                    FileUtils.copyFile(file, targetFile);
                }
                Files.createDirectories(infoDir, new FileAttribute[0]);
                Files.writeString(infoFile, (CharSequence)("[Trash Info]\nPath=" + FileUtils.getAbsolutePath(file) + "\nDeletionDate=" + time + "\n"), new OpenOption[0]);
                FileUtils.forceDelete(file);
            }
            catch (IOException e) {
                Logger.LOG.warning("Failed to move " + String.valueOf(file) + " to trash", e);
                return false;
            }
            return true;
        }
        try {
            return Desktop.getDesktop().moveToTrash(file.toFile());
        }
        catch (Exception e) {
            return false;
        }
    }

    public static void cleanDirectory(final Path directory) throws IOException {
        if (!Files.exists(directory, new LinkOption[0])) {
            Files.createDirectories(directory, new FileAttribute[0]);
            return;
        }
        if (!Files.isDirectory(directory, new LinkOption[0])) {
            String message = String.valueOf(directory) + " is not a directory";
            throw new IllegalArgumentException(message);
        }
        Files.walkFileTree(directory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            @NotNull
            public FileVisitResult visitFile(@NotNull Path file, @NotNull BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            @NotNull
            public FileVisitResult postVisitDirectory(@NotNull Path dir, @Nullable IOException exc) throws IOException {
                if (!dir.equals(directory)) {
                    Files.delete(dir);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    @CanIgnoreReturnValue
    public static boolean cleanDirectoryQuietly(Path directory) {
        try {
            FileUtils.cleanDirectory(directory);
            return true;
        }
        catch (IOException e) {
            return false;
        }
    }

    public static void forceDelete(Path file) throws IOException {
        if (Files.isDirectory(file, new LinkOption[0])) {
            FileUtils.deleteDirectory(file);
        } else {
            Files.delete(file);
        }
    }

    public static void copyFile(Path srcFile, Path destFile) throws IOException {
        Objects.requireNonNull(srcFile, "Source must not be null");
        Objects.requireNonNull(destFile, "Destination must not be null");
        if (!Files.exists(srcFile, new LinkOption[0])) {
            throw new FileNotFoundException("Source '" + String.valueOf(srcFile) + "' does not exist");
        }
        if (Files.isDirectory(srcFile, new LinkOption[0])) {
            throw new IOException("Source '" + String.valueOf(srcFile) + "' exists but is a directory");
        }
        Files.createDirectories(destFile.getParent(), new FileAttribute[0]);
        if (Files.exists(destFile, new LinkOption[0]) && !Files.isWritable(destFile)) {
            throw new IOException("Destination '" + String.valueOf(destFile) + "' exists but is read-only");
        }
        Files.copy(srcFile, destFile, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
    }

    public static List<Path> listFilesByExtension(Path file, String extension) {
        List<Path> list;
        block8: {
            Stream<Path> list2 = Files.list(file);
            try {
                list = list2.filter(it -> Files.isRegularFile(it, new LinkOption[0]) && extension.equals(FileUtils.getExtension(it))).toList();
                if (list2 == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (list2 != null) {
                        try {
                            list2.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    Logger.LOG.warning("Failed to list files by extension " + extension, e);
                    return List.of();
                }
            }
            list2.close();
        }
        return list;
    }

    public static Optional<Path> tryGetPath(String first, String ... more) {
        if (first == null) {
            return Optional.empty();
        }
        try {
            return Optional.of(Paths.get(first, more));
        }
        catch (InvalidPathException e) {
            return Optional.empty();
        }
    }

    public static Path tmpSaveFile(Path file) {
        return file.toAbsolutePath().resolveSibling("." + file.getFileName().toString() + ".tmp");
    }

    public static void saveSafely(Path file, String content) throws IOException {
        Path tmpFile = FileUtils.tmpSaveFile(file);
        try (BufferedWriter writer = Files.newBufferedWriter(tmpFile, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE);){
            writer.write(content);
        }
        try {
            if (Files.exists(file, new LinkOption[0]) && Files.getAttribute(file, "dos:hidden", new LinkOption[0]) == Boolean.TRUE) {
                Files.setAttribute(tmpFile, "dos:hidden", true, new LinkOption[0]);
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        Files.move(tmpFile, file, StandardCopyOption.REPLACE_EXISTING);
    }

    public static void saveSafely(Path file, ExceptionalConsumer<? super OutputStream, IOException> action) throws IOException {
        Path tmpFile = FileUtils.tmpSaveFile(file);
        try (OutputStream os = Files.newOutputStream(tmpFile, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.CREATE);){
            action.accept(os);
        }
        try {
            if (Files.exists(file, new LinkOption[0]) && Files.getAttribute(file, "dos:hidden", new LinkOption[0]) == Boolean.TRUE) {
                Files.setAttribute(tmpFile, "dos:hidden", true, new LinkOption[0]);
            }
        }
        catch (Throwable throwable) {
            // empty catch block
        }
        Files.move(tmpFile, file, StandardCopyOption.REPLACE_EXISTING);
    }

    public static String printFileStructure(Path path, int maxDepth) throws IOException {
        return DirectoryStructurePrinter.list(path, maxDepth);
    }
}

